from collections import deque
from typing import Dict
from math import floor

import bpy
from . import state
from . import utility
from ..panels.remapping import TARGET_BLENDSHAPES

class Recorder():
    def __init__(self):
        self.queue = deque([])

    def store(self, packet):
        self.queue.append(packet)

    def bake(self):
        target = bpy.context.scene.dollars_target_armature
    
        # Check if target armature exists
        has_armature = target is not None and target
        armature_action = None
    
        # Only create armature_action if there is an armature
        if has_armature:
            armature_action = bpy.data.actions.new(name="Recorded: " + target.name)
            if armature_action:
                armature_action.use_fake_user = True
                target.animation_data_create().action = armature_action
                
                # Set action slot for Blender 4.4+
                if bpy.app.version >= (4, 4, 0):
                    try:
                        if hasattr(armature_action, "slots") and len(armature_action.slots) == 0:
                            armature_action.slots.new(id_type='OBJECT', name=target.name)
                        if hasattr(target.animation_data, "action_slot") and len(armature_action.slots) > 0:
                            target.animation_data.action_slot = armature_action.slots[0]
                    except:
                        pass

        mesh_actions = {}
        has_targets = False
    
        # Iterate through all targets in mesh_collection
        mesh_collection = bpy.context.scene.dollars_mesh_collection
        for item in mesh_collection.items:
            obj = item.mesh_object
            if obj and obj.name and bpy.data.objects.get(obj.name):
                has_targets = True
                hierarchy = self._return_hierarchy(bpy.data.objects.get(obj.name))
                for obj in hierarchy:
                    if obj.type == "MESH" and hasattr(obj.data, 'shape_keys') and obj.data.shape_keys:
                        mesh_action = bpy.data.actions.new(name="Recorded: " + obj.name)
                        if mesh_action:
                            mesh_action.use_fake_user = True
                            obj.data.shape_keys.animation_data_create().action = mesh_action
                            mesh_actions[obj.name] = mesh_action
                            
                            # Set action slot for Blender 4.4+
                            if bpy.app.version >= (4, 4, 0):
                                try:
                                    if hasattr(mesh_action, "slots") and len(mesh_action.slots) == 0:
                                        mesh_action.slots.new(id_type='KEY', name=obj.name)
                                    if hasattr(obj.data.shape_keys.animation_data, "action_slot") and len(mesh_action.slots) > 0:
                                        obj.data.shape_keys.animation_data.action_slot = mesh_action.slots[0]
                                except:
                                    pass

        data_root = deque([])
        data_bone = deque([])
        data_shapekey = deque([])

        # If queue is empty, return
        if not self.queue:
            return
        
        firsttime = self.queue[0]["timestamp"]
        while self.queue:
            item = self.queue.popleft()
            item["frame_number"] = floor(
                (item["timestamp"] - firsttime).total_seconds() * bpy.context.scene.render.fps) + 1

            if has_armature and item["address"] == "/Dollars/Ext/Root/Pos":
                data_root.append(item)
            if has_armature and item["address"] == "/Dollars/Ext/Bone/Pos":
                data_bone.append(item)
            if has_targets and (item["address"] == "/Dollars/Ext/Blend/Val" or item["address"] == "/Dollars/Ext/Blend/Apply"):
                data_shapekey.append(item)

        # Only process armature animation if armature exists
        if has_armature:
            self.bake_root(armature_action, data_root)
            self.bake_bone(armature_action, data_bone)
    
        # Only process shape key animation if target objects exist
        if has_targets and mesh_actions:
            self.bake_shapekeys(mesh_actions, data_shapekey)

    def bake_root(self, action: bpy.types.Action, data: deque):
        action.groups.new("Object Transforms")
        data_paths = {}
        while data:
            item = data.popleft()
            pose = utility.convert_root(item["args"])

            for path, values in pose.items():
                if not data_paths.get(path):
                    data_paths[path] = []
                for i, value in enumerate(values):
                    if len(data_paths[path]) < i + 1:
                        data_paths[path].append([])
                    frame = {
                        "frame_number": item["frame_number"],
                        "value": value
                    }
                    if len(data_paths[path][i]) == 0:
                        data_paths[path][i].append(frame)
                    if data_paths[path][i][len(data_paths[path][i]) - 1]["frame_number"] == item["frame_number"]:
                        data_paths[path][i][len(
                            data_paths[path][i]) - 1] = frame
                    else:
                        data_paths[path][i].append(frame)

        for data_path, component_lists in data_paths.items():
            for i, component_list in enumerate(component_lists):
                curve = action.fcurves.new(
                    data_path=data_path,
                    index=i,
                    action_group="Object Transforms"
                )
                curve.color_mode = "AUTO_RGB" if len(
                    component_lists) == 3 else "AUTO_YRGB"
                keyframe_points = curve.keyframe_points
                keyframe_points.add(len(component_list))
                for j, component in enumerate(component_list):
                    keyframe_points[j].co = (
                        component["frame_number"],
                        component["value"]
                    )
                    keyframe_points[j].interpolation = "LINEAR"

    def bake_bone(self, action: bpy.types.Action, data: deque):
        data_paths = {}
        while data:
            item = data.popleft()
            bone = state.config.humanoid_to_bone.get(item["args"][0])
            if bone is None:
                continue
            pose = utility.convert_pose(item["args"])

            for path, values in pose.items():
                bone_name = bone.pose_bone.name

                if (path == "location") and (item["args"][0] == "Hips"):
                    bone_name = state.config.humanoid_to_bone["root"].pose_bone.name

                if not data_paths.get(bone_name):
                    data_paths[bone_name] = {}
                if not data_paths[bone_name].get(path):
                    data_paths[bone_name][path] = []
                for i, value in enumerate(values):
                    if len(data_paths[bone_name][path]) < i + 1:
                        data_paths[bone_name][path].append([])
                    frame = {
                        "frame_number": item["frame_number"],
                        "value": value
                    }
                    if len(data_paths[bone_name][path][i]) == 0:
                        data_paths[bone_name][path][i].append(frame)
                    if data_paths[bone_name][path][i][len(data_paths[bone_name][path][i]) - 1]["frame_number"] == item["frame_number"]:
                        data_paths[bone_name][path][i][len(
                            data_paths[bone_name][path][i]) - 1] = frame
                    else:
                        data_paths[bone_name][path][i].append(frame)

        for bone_name, paths in data_paths.items():
            action.groups.new(bone_name)
            for path, component_lists in paths.items():
                for i, component_list in enumerate(component_lists):
                    data_path = "pose.bones[\"" + bone_name + "\"]." + path
                    curve = action.fcurves.new(
                        data_path=data_path,
                        index=i,
                        action_group=bone_name
                    )
                    curve.color_mode = "AUTO_RGB" if len(
                        component_lists) == 3 else "AUTO_YRGB"
                    keyframe_points = curve.keyframe_points
                    keyframe_points.add(len(component_list))
                    for j, component in enumerate(component_list):
                        keyframe_points[j].co = (
                            component["frame_number"],
                            component["value"]
                        )
                        keyframe_points[j].interpolation = "LINEAR"

    def bake_shapekeys(self, actions: Dict[str, bpy.types.Action], data: deque):
        self.shapekey_queue = {}
        
        # Initialize storage for all target objects' shape key data
        self.all_targets_data_paths = {}
        
        # Get all shape keys from all target objects in mesh_collection
        mesh_collection = bpy.context.scene.dollars_mesh_collection
        for item in mesh_collection.items:
            obj = item.mesh_object
            if obj and obj.name and bpy.data.objects.get(obj.name):
                hierarchy = self._return_hierarchy(bpy.data.objects.get(obj.name))
                for obj in hierarchy:
                    if obj.type == "MESH" and hasattr(obj.data, 'shape_keys') and obj.data.shape_keys:
                        if obj.name not in self.all_targets_data_paths:
                            self.all_targets_data_paths[obj.name] = {}
        
        while data:
            item = data.popleft()
            if item["address"] == "/Dollars/Ext/Blend/Apply":
                self._apply_shapekey(actions, item["frame_number"])
            else:
                self._accumulate_shapekey(item["args"])
        
        # Create keyframes for all shape keys
        for obj_name, shapekeys in self.all_targets_data_paths.items():
            if obj_name in actions:
                action = actions[obj_name]
                for shapekey_name, frames in shapekeys.items():
                    data_path = "key_blocks[\"" + shapekey_name + "\"].value"
                    curve = action.fcurves.new(data_path=data_path)
                    keyframe_points = curve.keyframe_points
                    keyframe_points.add(len(frames))
                    for i, frame in enumerate(frames):
                        keyframe_points[i].co = (
                            frame["frame_number"],
                            frame["value"]
                        )
                        keyframe_points[i].interpolation = "LINEAR"

    def _apply_shapekey(self, actions, frame_number):
        # Process all targets in mesh_collection
        mesh_collection = bpy.context.scene.dollars_mesh_collection
        for item in mesh_collection.items:
            obj = item.mesh_object
            if obj and obj.name and bpy.data.objects.get(obj.name):
                hierarchy = self._return_hierarchy(bpy.data.objects.get(obj.name))
            
                for obj in hierarchy:
                    if obj.type == "MESH" and hasattr(obj.data, 'shape_keys') and obj.data.shape_keys:
                        mesh_name = obj.name
                        
                        # Create keyframes for this mesh's shape keys
                        for key_name in obj.data.shape_keys.key_blocks.keys():
                            value = 0.0
                            has_valid_mapping = False
                            
                            # Find corresponding value through remapping
                            for blendshape_name, queue_value in self.shapekey_queue.items():
                                try:
                                    # Try to get remapped shape key name
                                    index = TARGET_BLENDSHAPES.index(blendshape_name)
                                    remapped_name = bpy.context.scene.blendshape_remap_d[index].source
                                    
                                    # Check if mapping is valid (not empty)
                                    if remapped_name and key_name == remapped_name:
                                        value = queue_value
                                        has_valid_mapping = True
                                        break
                                except (ValueError, IndexError, AttributeError) as e:
                                    # Failed to find mapping, continue to next shape key
                                    continue
                            
                            # Only record frame data for shape keys with valid mapping
                            if has_valid_mapping:
                                # Store frame data
                                if mesh_name not in self.all_targets_data_paths:
                                    self.all_targets_data_paths[mesh_name] = {}
                                if key_name not in self.all_targets_data_paths[mesh_name]:
                                    self.all_targets_data_paths[mesh_name][key_name] = []
                            
                                key_sequence = self.all_targets_data_paths[mesh_name][key_name]
                                frame = {
                                    "frame_number": frame_number,
                                    "value": value
                                }
                            
                                # Add or update keyframe
                                if len(key_sequence) == 0:
                                    key_sequence.append(frame)
                                elif key_sequence[-1]["frame_number"] == frame["frame_number"]:
                                    key_sequence[-1] = frame
                                else:
                                    key_sequence.append(frame)
    
        # Clear queue
        self.shapekey_queue = {}

    def _accumulate_shapekey(self, args):
        if args[1] == 0.0:
            return
        blendshape_name = args[0]
        value = args[1]
        
        if blendshape_name not in self.shapekey_queue:
            self.shapekey_queue[blendshape_name] = 0.0
        self.shapekey_queue[blendshape_name] += value

    def _return_hierarchy(self, obj):
        hierarchy = [obj]
        for child in obj.children:
            hierarchy.extend(self._return_hierarchy(child))
        return hierarchy