import bpy
from bpy.types import (Operator,
                      PropertyGroup,
                      UIList)
from bpy.props import (IntProperty,
                      BoolProperty,
                      StringProperty,
                      CollectionProperty,
                      PointerProperty)

class MeshItem(PropertyGroup):
    mesh_object: PointerProperty(
        type=bpy.types.Object,
        name="Mesh Object",
        description="Select a Mesh object",
        poll=lambda self, obj: obj.type == 'MESH')

class MeshCollection(PropertyGroup):
    items: CollectionProperty(type=MeshItem)
    active_index: IntProperty(name="Active Index", default=0)

class MESH_OT_add_item(Operator):
    bl_idname = "dollars_mesh_list.add_item"
    bl_label = "Add Mesh"
    bl_description = "Add a new mesh to the list"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        mesh_collection = context.scene.dollars_mesh_collection
        item = mesh_collection.items.add()
        item.name = f"Mesh {len(mesh_collection.items)}"
        mesh_collection.active_index = len(mesh_collection.items) - 1
        return {'FINISHED'}

class MESH_OT_remove_item(Operator):
    bl_idname = "dollars_mesh_list.remove_item"
    bl_label = "Remove Mesh"
    bl_description = "Remove selected mesh from the list"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.scene.dollars_mesh_collection.items

    def execute(self, context):
        mesh_collection = context.scene.dollars_mesh_collection
        mesh_collection.items.remove(mesh_collection.active_index)
        
        if mesh_collection.active_index >= len(mesh_collection.items):
            mesh_collection.active_index = max(0, len(mesh_collection.items) - 1)
            
        return {'FINISHED'}

class MESH_OT_clear_list(Operator):
    bl_idname = "dollars_mesh_list.clear_list"
    bl_label = "Clear List"
    bl_description = "Clear the entire mesh list"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.scene.dollars_mesh_collection.items

    def execute(self, context):
        if bpy.ops.object.mode_set.poll():
            bpy.ops.object.mode_set(mode='OBJECT')
            
        mesh_collection = context.scene.dollars_mesh_collection
        mesh_collection.items.clear()
        mesh_collection.active_index = 0
        
        return {'FINISHED'}

class MESH_UL_items(UIList):
    def draw_item(self, context, layout, data, item, icon, active_data, active_propname, index):
        if self.layout_type in {'DEFAULT', 'COMPACT'}:
            row = layout.row(align=True)
            if item.mesh_object:
                row.label(text=item.mesh_object.name, icon='MESH_DATA')
            else:
                row.label(text="No Mesh Object Selected", icon='MESH_DATA')                
        elif self.layout_type in {'GRID'}:
            layout.alignment = 'CENTER'
            layout.label(text=item.name)

def register():
    bpy.utils.register_class(MeshItem)
    bpy.utils.register_class(MeshCollection)
    bpy.utils.register_class(MESH_OT_add_item)
    bpy.utils.register_class(MESH_OT_remove_item)
    bpy.utils.register_class(MESH_OT_clear_list)
    bpy.utils.register_class(MESH_UL_items)
    
    bpy.types.Scene.dollars_mesh_collection = PointerProperty(type=MeshCollection)

def unregister():
    del bpy.types.Scene.dollars_mesh_collection
    
    bpy.utils.unregister_class(MESH_UL_items)
    bpy.utils.unregister_class(MESH_OT_clear_list)
    bpy.utils.unregister_class(MESH_OT_remove_item)
    bpy.utils.unregister_class(MESH_OT_add_item)
    bpy.utils.unregister_class(MeshCollection)
    bpy.utils.unregister_class(MeshItem)