import bpy
from bpy import context
from bpy.types import Panel, UIList, PropertyGroup
from bpy.props import StringProperty
from ..core import state
from ..core import config
from bpy.app.handlers import persistent

# List of target blendshapes
TARGET_BLENDSHAPES = [
    "EyeBlinkLeft", "EyeLookDownLeft", "EyeLookInLeft", "EyeLookOutLeft", "EyeLookUpLeft",
    "EyeSquintLeft", "EyeWideLeft", "EyeBlinkRight", "EyeLookDownRight", "EyeLookInRight",
    "EyeLookOutRight", "EyeLookUpRight", "EyeSquintRight", "EyeWideRight", "JawForward",
    "JawRight", "JawLeft", "JawOpen", "MouthClose", "MouthFunnel", "MouthPucker",
    "MouthRight", "MouthLeft", "MouthSmileLeft", "MouthSmileRight", "MouthFrownLeft",
    "MouthFrownRight", "MouthDimpleLeft", "MouthDimpleRight", "MouthStretchLeft",
    "MouthStretchRight", "MouthRollLower", "MouthRollUpper", "MouthShrugLower",
    "MouthShrugUpper", "MouthPressLeft", "MouthPressRight", "MouthLowerDownLeft",
    "MouthLowerDownRight", "MouthUpperUpLeft", "MouthUpperUpRight", "BrowDownLeft",
    "BrowDownRight", "BrowInnerUp", "BrowOuterUpLeft", "BrowOuterUpRight", "CheekPuff",
    "CheekSquintLeft", "CheekSquintRight", "NoseSneerLeft", "NoseSneerRight", "TongueOut",
    "HeadYaw", "HeadPitch", "HeadRoll"
]

class BlendshapeRemapItem(PropertyGroup):
    source: StringProperty(name="Source Blendshape")

class BLENDSHAPE_UL_remap_list_D(UIList):
    def draw_item(self, context, layout, data, item, icon, active_data, active_propname, index):
        if self.layout_type in {'DEFAULT', 'COMPACT'}:
            row = layout.row(align=True)
            row.label(text=TARGET_BLENDSHAPES[index])
            row.prop(item, "source", text="", emboss=False)

@persistent
def initialize_blendshape_remap(dummy):
    # Check if we need to initialize
    if len(bpy.context.scene.blendshape_remap_d) == len(TARGET_BLENDSHAPES) - 3:
        return
    
    # Clear the collection and add the items
    bpy.context.scene.blendshape_remap_d.clear()
    for target in TARGET_BLENDSHAPES[:-3]:  # Excluding HeadYaw, HeadPitch, HeadRoll
        item = bpy.context.scene.blendshape_remap_d.add()
        item.source = target[0].lower() + target[1:]  # Default to camelCase versions

class RemappingPanel(bpy.types.Panel):
    bl_label = "Blendshape and Bone Remapping"
    bl_idname = "OBJECT_PT_DOLLARSMOCAP_BONES"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = 'DollarsMoCap'
    bl_options = {'DEFAULT_CLOSED'}
    
    @classmethod
    def poll(cls, context: context):
        return True
        
    def draw_header(self, context):
        try:
            layout = self.layout
            row = layout.row(align=True)
            
            # Add save/load buttons
            row.operator("dollarsmocap.load_mappings", text="Import", icon='IMPORT')
            row.operator("dollarsmocap.save_mappings", text="Export", icon='EXPORT')
            
        except Exception as exc:
            print(str(exc) + " | Error in Dollars panel header")

    def draw(self, context):
        layout = self.layout

        # Blendshape section
        layout.label(text="Blendshape Remapping")
        box_blendshapes = layout.box()
        row = box_blendshapes.row()
        row.label(text="ARKit Blendshape")
        row.label(text="Target Blendshape")
        row = box_blendshapes.row()
        row.template_list("BLENDSHAPE_UL_remap_list_D", "", context.scene, "blendshape_remap_d", 
                          context.scene, "blendshape_remap_index_d", rows=10)
        
        # Bone sections
        layout.label(text="Torso Bones")
        box_torso = layout.box()
        layout.label(text="Left Hand Bones")
        box_left_hand = layout.box()
        layout.label(text="Right Hand Bones")
        box_right_hand = layout.box()
        
        if state.server is not None and state.server.is_alive():
            box_torso.active = False
            box_right_hand.active = False
            box_left_hand.active = False
        
        # Bone mappings
        if not bpy.context.scene.dollars_target_armature:
            return
        
        armature_obj = context.scene.dollars_target_armature
        for bone in config.human_body_bones_torso:
            box_torso.prop_search(
                context.scene,
                "dollars_bones_" + bone,
                armature_obj.pose,
                "bones"
            )
        for bone in config.human_body_bones_right_hand:
            box_right_hand.prop_search(
                context.scene,
                "dollars_bones_" + bone,
                armature_obj.pose,
                "bones"
            )
        for bone in config.human_body_bones_left_hand:
            box_left_hand.prop_search(
                context.scene,
                "dollars_bones_" + bone,
                armature_obj.pose,
                "bones"
            )
            
def register():
    # Register classes
    bpy.utils.register_class(BlendshapeRemapItem)
    bpy.utils.register_class(BLENDSHAPE_UL_remap_list_D)
    
    # Register properties
    bpy.types.Scene.blendshape_remap_d = bpy.props.CollectionProperty(type=BlendshapeRemapItem)
    bpy.types.Scene.blendshape_remap_index_d = bpy.props.IntProperty()
    
    bpy.app.handlers.load_post.append(initialize_blendshape_remap)
    # Initialize remap data
    bpy.app.timers.register(lambda: initialize_blendshape_remap(None), first_interval=0.1)

def unregister():
    # Unregister classes
    bpy.utils.unregister_class(BlendshapeRemapItem)
    bpy.utils.unregister_class(BLENDSHAPE_UL_remap_list_D)
