import bpy
from .bone_candidates import bone_list
from .config import human_body_bones_torso, human_body_bones_left_hand, human_body_bones_right_hand

def standardize_bone_name(name):
    # List of chars to replace if they are at the start of a bone name
    starts_with = [
        ('_', ''),
        ('ValveBiped_', ''),
        ('Valvebiped_', ''),
        ('Bip1_', 'Bip_'),
        ('Bip01_', 'Bip_'),
        ('Bip001_', 'Bip_'),
        ('Character1_', ''),
        ('HLP_', ''),
        ('JD_', ''),
        ('JU_', ''),
        ('Armature|', ''),
        ('Bone_', ''),
        ('C_', ''),
        ('Cf_S_', ''),
        ('Cf_J_', ''),
        ('G_', ''),
        ('Joint_', ''),
        ('DEF_', ''),
    ]

    # Standardize names
    # Make all the underscores!
    name = name.replace(' ', '_') \
        .replace('-', '_') \
        .replace('.', '_') \
        .replace('____', '_') \
        .replace('___', '_') \
        .replace('__', '_') \

    # Replace if name starts with specified chars
    for replacement in starts_with:
        if name.startswith(replacement[0]):
            name = replacement[1] + name[len(replacement[0]):]

    # Remove digits from the start
    name_split = name.split('_')
    if len(name_split) > 1 and name_split[0].isdigit():
        name = name_split[1]

    # Specific condition
    name_split = name.split('"')
    if len(name_split) > 3:
        name = name_split[1]

    # Another specific condition
    if ':' in name:
        for i, split in enumerate(name.split(':')):
            if i == 0:
                name = ''
            else:
                name += split

    # Remove S0 from the end
    if name[-2:] == 'S0':
        name = name[:-2]

    if name[-4:] == '_Jnt':
        name = name[:-4]
    return name.lower()


def detect_bone(armature, bone_name_key):
    found_bone_name = ''
    
    # Get the list of possible bone names for this key
    possible_bone_names = []
    
    # Handle Left/Right cases
    lookup_key = bone_name_key
    if "Left" in bone_name_key:
        # Use the original bone_name_key to look up in bone_list
        if bone_name_key in bone_list:
            possible_bone_names = [name.lower() for name in bone_list[bone_name_key]]
            # Replace '\l' with 'l' and 'left'
            modified_names = []
            for name in possible_bone_names:
                modified_names.append(name.replace('\\l', 'l'))
                modified_names.append(name.replace('\\l', 'left'))
            possible_bone_names = modified_names
    elif "Right" in bone_name_key:
        # Replace "Right" with "Left" to look up in bone_list
        left_key = bone_name_key.replace("Right", "Left")
        if left_key in bone_list:
            possible_bone_names = [name.lower() for name in bone_list[left_key]]
            # Replace '\l' with 'r' and 'right'
            modified_names = []
            for name in possible_bone_names:
                modified_names.append(name.replace('\\l', 'r'))
                modified_names.append(name.replace('\\l', 'right'))
            possible_bone_names = modified_names
    # Standard case - direct lookup
    elif bone_name_key in bone_list:
        possible_bone_names = [name.lower() for name in bone_list[bone_name_key]]

    # Go through all bones in the armature
    for bone_name_candidate in possible_bone_names:
        for bone in armature.pose.bones:
            if bone_name_candidate.lower() == standardize_bone_name(bone.name):
                found_bone_name = bone.name
                return found_bone_name
                
    # Fallback: check if bone_name_key is part of any bone name
    for bone in armature.pose.bones:
        if bone_name_key.lower() in standardize_bone_name(bone.name):
            found_bone_name = bone.name
            return found_bone_name
            
    return found_bone_name

def auto_map_bones(self, context):
    armature = self.dollars_target_armature
    if not armature:
        return    

    if armature.type != 'ARMATURE':
        return
    
    # Combine all humanoid bones from config.py
    all_human_bones = [
        *human_body_bones_torso,
        *human_body_bones_left_hand,
        *human_body_bones_right_hand
    ]
    
    
    # Store which bones have already been mapped to avoid duplicates
    mapped_bones = set()
    
    # Process special bones (UpperChest, Chest) with reverse lookup first
    spine_special_bones = ['UpperChest', 'Chest']
    for bone_name in spine_special_bones:
        if bone_name in all_human_bones:
            # Apply reverse lookup for Spine bones list
            reversed_spine_list = list(reversed(bone_list['Spine']))
            
            # Try to find a match in the armature for each potential bone name
            for spine_bone_name in reversed_spine_list:
                found = False
                for bone in armature.pose.bones:
                    if spine_bone_name.lower() == standardize_bone_name(bone.name):
                        # Found a match
                        if bone.name not in mapped_bones:
                            setattr(context.scene, f"dollars_bones_{bone_name}", bone.name)
                            mapped_bones.add(bone.name)
                            found = True
                            break
                if found:
                    break
    
    # Process the rest of the bones normally
    for bone_name in all_human_bones:
        # Skip the special bones we already processed
        if bone_name in spine_special_bones:
            continue
            
        # Try to detect the corresponding bone in the armature
        detected_bone = detect_bone(armature, bone_name)
        
        # If a bone was found and has not been mapped yet
        if detected_bone and detected_bone not in mapped_bones:
            setattr(context.scene, f"dollars_bones_{bone_name}", detected_bone)
            mapped_bones.add(detected_bone)
        else:
            # Clear the property if no bone was found or it's already mapped
            if not detected_bone:
                setattr(context.scene, f"dollars_bones_{bone_name}", "")

def on_target_armature_update(self, context):
    armature_obj = self.dollars_target_armature
    if armature_obj:
        auto_map_bones(self, context)

def poll_armature(self, obj):
    return obj.type == 'ARMATURE'
